<?php

namespace App\Http\Controllers\Frontend\Profile;

use App\Enums\PostStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\ProductRequest;
use App\Http\Requests\ProfileUpdateRequest;
use App\Models\Setting;
use App\Models\Wishlist;
use App\Repositories\BoostPlanRepository;
use App\Repositories\BoostPostRepository;
use App\Repositories\SellingPostRepository;
use App\Repositories\LanguageRepository;
use App\Repositories\NotificationHistoryRepository;
use App\Repositories\ReviewRepository;
use App\Repositories\PaymentGatewayRepository;
use App\Repositories\TransactionRepository;
use App\Repositories\UserRepository;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;

class ProfileController extends Controller
{
    //Show Profile Page with data
    public function showProfile()
    {
        $myProfileData = UserRepository::query()->with('profilePhoto')->find(auth()->id());
        $languages = LanguageRepository::getAll()->where('status','1');
        return view('web.profile.sections.profile', compact('myProfileData', 'languages'));
    }


    public function changeLanguage($language)
    {
        $availableLanguages = LanguageRepository::getAll()->pluck('name')->toArray();

        if (in_array($language, $availableLanguages)) {
            session(['locale' => $language]);
            app()->setLocale($language);
        }

        return redirect()->back();
    }



    public function update(ProfileUpdateRequest $request, $id)
    {
        UserRepository::updateProfile($request, $id);
        return redirect()->back()->with('success', 'Profile updated successfully!');
    }


    public function showMyAds(Request $request)
    {
        $wishlistIds = Wishlist::where('user_id', auth()->id())
            ->pluck('selling_post_id')
            ->toArray();

        $query = SellingPostRepository::query()
            ->where('user_id', auth()->id())
            ->with(['user', 'brand', 'thumbnails', 'activeBoost']);


        if ($request->has('all')) {
            $products = $query->get();
        } else {
            $products = $query->take(6)->get();
        }

        return view('web.profile.sections.my-ads', compact('products', 'wishlistIds'));
    }

    public function showWishlist()
    {
        // Fetch wishlist IDs
        $wishlistIds = Wishlist::where('user_id', auth()->id())->pluck('selling_post_id')->toArray();

        // Fetch products in wishlist
        $products = SellingPostRepository::query()
            ->whereIn('id', $wishlistIds)
            ->with(['user', 'brand', 'thumbnails'])
            ->paginate(6);
        return view('web.profile.sections.wishlist', compact('products', 'wishlistIds'));
    }


    public function showUserNotifications()
    {
        $notifications = getNotifications(true);
        return view('web.profile.sections.notification', compact('notifications'));
    }


    public function readSingleNotification(Request $request)
    {
        $notification = NotificationHistoryRepository::findOrFail($request->notification_id);
        $notification->update(['is_read' => true]);

        return response()->json([
            'status' => 'success',
            'redirect_url' => route('user.profile.show', $notification->sender_id),
        ]);
    }

    //Mak all notifications is_read ture
    public function readAllNotifications()
    {
        $id = auth()->id();
        NotificationHistoryRepository::query()->where('receiver_id', $id)->where('is_read', false)->update(['is_read' => true]);
        session()->flash('success', 'All notifications marked as read.');
        return redirect()->route('user.show-notifications');
    }

    public function reviewIndex()
    {
        $reviewes = ReviewRepository::query()->where('seller_id', auth()->id())->with('reviewer')->paginate(6);
        return view('web.profile.sections.review', compact('reviewes'));
    }

    public function moveToTrash($id)
    {

        $post = SellingPostRepository::query()->where('user_id', auth()->id())->findOrFail($id);

        // Soft delete
        $post->delete();

        return back()->with('success', 'Post moved to trash!');
    }


    public function trashIndex()
    {
        $products = SellingPostRepository::trashedByUser(auth()->id())->get();
        return view('web.profile.sections.trash', compact('products'));
    }
    public function restore($id)
    {
        SellingPostRepository::restoreByUser($id, auth()->id());
        return back()->with('success', 'Post restored successfully!');
    }


    public function showPromoteAds($id)
    {
        $product = SellingPostRepository::query()->where('id', $id)
            ->where('user_id', auth()->id())
            ->firstOrFail();
        $boostPlans = BoostPlanRepository::getAll();

        return view('web.profile.sections.promot-ads', compact('product', 'boostPlans'));
    }


    public function soldout($id)
    {
        $product = SellingPostRepository::find($id);
        if (!$product) {
            return redirect()->back()->with('error', 'Product not found!');
        }
        $product->status = 'Soled';
        $product->save();

        return redirect()->back()->with('success', 'Product marked as Sold Out!');
    }

    public function soldOutAds(Request $request)
    {

        $userId = auth()->id();

        $soldOutAds = SellingPostRepository::query()
            ->where('user_id', $userId)
            ->where('status', 'Soled')
            ->latest()
            ->get();

        return view('web.profile.sections.sold-out', compact('soldOutAds'));
    }

    public function markAsSold(Request $request)
    {

        $product = SellingPostRepository::findOrFail($request->product_id);
        $product->update([
            'status' => PostStatus::Soled->value,
            'sold_price' => $request->sold_price,
            'buyer_name' => $request->buyer_name,
            'is_sold' => true,
            'updated_at' => now(),
        ]);

        return back()->with('success', 'Product marked as sold successfully!');
    }
}
